/* Copyright (C) 2015-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef OPTIONAL_H_d8973500_1e52_40dd_9288_c792d1f89b0f
#define OPTIONAL_H_d8973500_1e52_40dd_9288_c792d1f89b0f


#include "OptionalReference.h"
#include <stdexcept>

/**
 * \cond VNCCOMMON
 */
namespace vnccommon
{

/**
 * \brief %Optional value.
 */
template <typename T>
class Optional
{
public:
    /**
     * \brief Default constructor.
     *
     * Creates an \ref Optional representing 'nothing'.
     */
    Optional()
        :   mValue(),
            mHasValue(false)
    {
    }

    /**
     * \brief 'Nothing' constructor.
     *
     * Creates an \ref Optional representing 'nothing'.
     *
     * \param none A 'nothing' value.
     */
    Optional(NoneType none)
        :   mValue(),
            mHasValue(false)
    {
        (void) none;
    }

    /**
     * \brief 'Something' constructor.
     *
     * Creates an \ref Optional representing 'something'.
     *
     * \param value The value the \ref Optional object will hold.
     */
    explicit Optional(const T& value)
        :   mValue(value),
            mHasValue(true)
    {
    }

    /**
     * \brief Copy constructor.
     *
     * \param o The \ref Optional to copy.
     */
    Optional(const Optional<T>& o)
        :   mValue(o.mValue),
            mHasValue(o.mHasValue)
    {
    }

    /**
     * \brief Assignment operator.
     *
     * \param o The \ref Optional to copy.
     *
     * \return This \ref Optional.
     */
    Optional<T>& operator=(const Optional<T>& o)
    {
        mValue = o.mValue;
        mHasValue = o.mHasValue;
        return *this;
    }

    /**
     * \brief Queries whether the \ref Optional has a value.
     *
     * \return Whether the \ref Optional has a value.
     */
    bool hasValue() const
    {
        return mHasValue;
    }

    /**
     * \brief Gets the \ref Optional's value.
     *
     * \return The \ref Optional's value.
     */
    const T& value() const
    {
        if (!mHasValue)
        {
            throw std::logic_error("Attempted to retrieve value of empty Optional");
        }
        return mValue;
    }

    /**
     * \brief Gets the \ref Optional's value.
     *
     * \return The \ref Optional's value.
     */
    T& value()
    {
        if (!mHasValue)
        {
            throw std::logic_error("Attempted to retrieve value of empty Optional");
        }
        return mValue;
    }

    /**
     * \brief Compare this \ref Optional with another.
     *
     * \param other The other \ref Optional to compare against.
     * \return Whether this \ref Optional is equal to \a other.
     */
    bool operator==(const Optional<T>& other) const
    {
        if (hasValue() != other.hasValue())
        {
            return false;
        }

        if (!hasValue())
        {
            return true;
        }
        else
        {
            return value() == other.value();
        }
    }

    /**
     * \brief Returns true iff this Optional contains a value, and that value
     * is equal to the one provided.
     */
    bool contains(const T& o) const
    {
        if(!hasValue())
        {
            return false;
        }

        return value() == o;
    }

    /**
     * \brief Returns an \ref OptionalReference to the value in this \ref Optional.
     *
     * \return An \ref OptionalReference.
     */
    OptionalReference<const T> getReference() const
    {
        if(hasValue())
        {
            return OptionalReference<const T>(value());
        }
        else
        {
            return OptionalReference<const T>();
        }
    }

    /**
     * \brief Returns an \ref OptionalReference to the value in this \ref Optional.
     *
     * \return An \ref OptionalReference.
     */
    OptionalReference<T> getReference()
    {
        if(hasValue())
        {
            return OptionalReference<T>(value());
        }
        else
        {
            return OptionalReference<T>();
        }
    }

    /**
     * \brief If this Optional has a value, then it is returned by this method.
     * Otherwise, the provided value is returned.
     */
    T& orElse(T& alternative)
    {
        if(hasValue())
        {
            return value();
        }
        else
        {
            return alternative;
        }
    }

    /**
     * \brief If this Optional has a value, then it is returned by this method.
     * Otherwise, the provided value is returned.
     */
    const T& orElse(const T& alternative) const
    {
        if(hasValue())
        {
            return value();
        }
        else
        {
            return alternative;
        }
    }

    /**
     * \brief Throws the specified exception if no value is present.
     */
    template<typename E>
    const T& orThrow(const E& e) const
    {
        if(!hasValue())
        {
            throw e;
        }

        return value();
    }

    /**
     * \brief Throws the specified exception if no value is present.
     */
    template<typename E>
    T& orThrow(const E& e)
    {
        if(!hasValue())
        {
            throw e;
        }

        return value();
    }

private:
    // Putting the bool last avoids wasting bytes on padding
    T mValue;
    bool mHasValue;
};

/**
 * \brief Convenience function for constructing an \ref Optional.
 *
 * \param value The value the \ref Optional object will hold.
 * \return An \ref Optional representing 'something'.
 */
template <typename T>
Optional<T> MakeOptional(T value)
{
    return Optional<T>(value);
}

} // end of namespace vnccommon
/**
 * \endcond
 */

#endif // OPTIONAL_H_d8973500_1e52_40dd_9288_c792d1f89b0f

